// Ultimate Auto Care - API Integration Layer

const API_BASE_URL = ''; // Local calls

const api = {
    async submit(endpoint, data) {
        // Map old endpoint names to local action parameters
        let action = endpoint.split('/').pop();
        if (action === 'subscribe') action = 'newsletter';
        if (action === 'apply') action = 'financing';

        try {
            const response = await fetch(`handler.php?action=${action}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                },
                body: JSON.stringify(data)
            });
            return await response.json();
        } catch (error) {
            console.error('API Error:', error);
            return { success: false, message: 'Connection error. Please try again.' };
        }
    },
    async get(endpoint) {
        // This is now mostly handled by PHP direct renders, but kept for compatibility
        return { success: false, message: 'Direct PHP rendering preferred.' };
    }
};

// Global notification helper
window.showNotification = (message, type = 'success') => {
    const toast = document.createElement('div');
    toast.className = `fixed bottom-8 right-8 z-[200] px-8 py-4 rounded-2xl glass border animate-in shadow-2xl flex items-center space-x-3`;

    if (type === 'success') {
        toast.classList.add('border-green-500/30', 'text-green-400');
        toast.innerHTML = `<i class="fas fa-check-circle"></i><span>${message}</span>`;
    } else {
        toast.classList.add('border-rose-500/30', 'text-rose-400');
        toast.innerHTML = `<i class="fas fa-exclamation-circle"></i><span>${message}</span>`;
    }

    document.body.appendChild(toast);
    setTimeout(() => {
        toast.style.opacity = '0';
        toast.style.transform = 'translateY(20px)';
        setTimeout(() => toast.remove(), 500);
    }, 5000);
};

// Form Initializations
document.addEventListener('DOMContentLoaded', () => {
    // 1. Contact Form
    const contactForm = document.querySelector('form[x-show="tab === \'contact\'"]');
    if (contactForm) {
        contactForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            const btn = contactForm.querySelector('button');
            const originalText = btn.innerHTML;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i> Sending...';
            btn.disabled = true;

            const formData = {
                name: contactForm.querySelector('input[placeholder="John Doe"]').value,
                email: contactForm.querySelector('input[type="email"]').value,
                subject: contactForm.querySelector('input[placeholder="How can we help?"]').value,
                message: contactForm.querySelector('textarea').value
            };

            const res = await api.submit('/contact', formData);

            if (res.success) {
                window.showNotification(res.message);
                contactForm.reset();
            } else {
                window.showNotification(res.errors ? Object.values(res.errors)[0][0] : res.message, 'error');
            }

            btn.innerHTML = originalText;
            btn.disabled = false;
        });
    }

    // 2. Booking Form
    const bookingForm = document.querySelector('form[x-show="tab === \'booking\'"]');
    if (bookingForm) {
        bookingForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            const btn = bookingForm.querySelector('button');
            const originalText = btn.innerHTML;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i> Processing...';
            btn.disabled = true;

            const formData = {
                customer_name: bookingForm.querySelector('input[placeholder="John Doe"]').value,
                customer_email: bookingForm.querySelector('input[type="email"]').value,
                customer_phone: bookingForm.querySelector('input[type="tel"]').value,
                vehicle_make: bookingForm.querySelector('input[placeholder="e.g. 1995 Honda Civic"]').value,
                service_type: bookingForm.querySelector('select').value,
                preferred_date: bookingForm.querySelector('input[type="datetime-local"]').value,
                notes: bookingForm.querySelector('textarea').value
            };

            const res = await api.submit('/booking', formData);

            if (res.success) {
                window.showNotification(res.message);
                bookingForm.reset();
            } else {
                window.showNotification(res.errors ? Object.values(res.errors)[0][0] : res.message, 'error');
            }

            btn.innerHTML = originalText;
            btn.disabled = false;
        });
    }

    // 3. Newsletter Form
    const newsletterForm = document.querySelector('footer form');
    if (newsletterForm) {
        newsletterForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            const input = newsletterForm.querySelector('input');
            const btn = newsletterForm.querySelector('button');
            const originalText = btn.innerHTML;

            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
            btn.disabled = true;

            const res = await api.submit('/newsletter/subscribe', { email: input.value });

            if (res.success) {
                window.showNotification(res.message);
                input.value = '';
            } else {
                window.showNotification(res.errors ? Object.values(res.errors)[0][0] : res.message, 'error');
            }

            btn.innerHTML = originalText;
            btn.disabled = false;
        });
    }
});



