<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Car;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class CarsController extends Controller
{
    public function index()
    {
        global $conn;
        $stmt = $conn->query("SELECT * FROM cars ORDER BY created_at DESC");
        $cars = array_map(fn($item) => (object)$item, $stmt->fetchAll());
        return view('admin.cars.index', compact('cars'));
    }

    public function create()
    {
        return view('admin.cars.create');
    }

    public function store(Request $request)
    {
        global $conn;
        $validated = $request->validate([
            'make' => 'required|string',
            'model' => 'required|string',
            'vin' => 'nullable|string',
            'stock_number' => 'nullable|string',
            'year' => 'required|integer',
            'price' => 'required|numeric',
            'market_avg_price' => 'nullable|numeric',
            'type' => 'required|string',
            'description' => 'nullable|string',
            'image' => 'nullable|image|max:2048',
            'image_alt' => 'required_with:image|nullable|string',
            'fuel_type' => 'nullable|string',
            'transmission' => 'nullable|string',
            'mileage' => 'nullable|string',
            'ext_color' => 'nullable|string',
            'int_color' => 'nullable|string',
            'engine' => 'nullable|string',
            'drive_type' => 'nullable|string',
            'is_featured' => 'boolean',
            'status' => 'required|string',
            'seller_name' => 'nullable|string',
            'slug' => 'nullable|string|unique:cars,slug',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
        ]);

        $image_path = null;
        if ($request->hasFile('image')) {
            $path = \App\Services\ImageService::uploadAndOptimize($request->file('image'), 'cars');
            $image_path = asset('storage/' . $path);
        }

        $is_featured = $request->has('is_featured') ? 1 : 0;

        $sql = "INSERT INTO cars (make, model, vin, stock_number, year, price, market_avg_price, type, description, image, image_alt, fuel_type, transmission, mileage, ext_color, int_color, engine, drive_type, is_featured, status, seller_name, slug, meta_title, meta_description, created_at, updated_at) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())";
        
        $stmt = $conn->prepare($sql);
        $stmt->execute([
            $validated['make'], $validated['model'], $validated['vin'] ?? '', $validated['stock_number'] ?? '', $validated['year'], 
            $validated['price'], $validated['market_avg_price'] ?? 0, $validated['type'], $validated['description'] ?? '', 
            $image_path, $validated['image_alt'] ?? '', $validated['fuel_type'] ?? '', $validated['transmission'] ?? '', 
            $validated['mileage'] ?? '', $validated['ext_color'] ?? '', $validated['int_color'] ?? '', 
            $validated['engine'] ?? '', $validated['drive_type'] ?? '', $is_featured, $validated['status'], 
            $validated['seller_name'] ?? '', $validated['slug'] ?? '', $validated['meta_title'] ?? '', $validated['meta_description'] ?? ''
        ]);

        return redirect()->route('admin.cars.index')->with('success', 'Car added successfully.');
    }

    public function edit($id)
    {
        global $conn;
        $stmt = $conn->prepare("SELECT * FROM cars WHERE id = ? LIMIT 1");
        $stmt->execute([$id]);
        $car = (object)$stmt->fetch();
        if (!$car || !isset($car->id)) abort(404);
        
        return view('admin.cars.edit', compact('car'));
    }

    public function update(Request $request, $id)
    {
        global $conn;
        $validated = $request->validate([
            'make' => 'required|string',
            'model' => 'required|string',
            'vin' => 'nullable|string',
            'stock_number' => 'nullable|string',
            'year' => 'required|integer',
            'price' => 'required|numeric',
            'market_avg_price' => 'nullable|numeric',
            'type' => 'required|string',
            'description' => 'nullable|string',
            'image' => 'nullable|image|max:2048',
            'image_alt' => 'required_with:image|nullable|string',
            'fuel_type' => 'nullable|string',
            'transmission' => 'nullable|string',
            'mileage' => 'nullable|string',
            'ext_color' => 'nullable|string',
            'int_color' => 'nullable|string',
            'engine' => 'nullable|string',
            'drive_type' => 'nullable|string',
            'is_featured' => 'sometimes|boolean',
            'status' => 'required|string',
            'seller_name' => 'nullable|string',
            'slug' => 'nullable|string', // Unique check skipped for brevity or use manual check
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
        ]);

        $sql = "UPDATE cars SET make=?, model=?, vin=?, stock_number=?, year=?, price=?, market_avg_price=?, type=?, description=?, image_alt=?, fuel_type=?, transmission=?, mileage=?, ext_color=?, int_color=?, engine=?, drive_type=?, is_featured=?, status=?, seller_name=?, slug=?, meta_title=?, meta_description=?, updated_at=NOW()";
        $params = [
            $validated['make'], $validated['model'], $validated['vin'] ?? '', $validated['stock_number'] ?? '', $validated['year'], 
            $validated['price'], $validated['market_avg_price'] ?? 0, $validated['type'], $validated['description'] ?? '', 
            $validated['image_alt'] ?? '', $validated['fuel_type'] ?? '', $validated['transmission'] ?? '', 
            $validated['mileage'] ?? '', $validated['ext_color'] ?? '', $validated['int_color'] ?? '', 
            $validated['engine'] ?? '', $validated['drive_type'] ?? '', $request->has('is_featured') ? 1 : 0, 
            $validated['status'], $validated['seller_name'] ?? '', $validated['slug'] ?? '', 
            $validated['meta_title'] ?? '', $validated['meta_description'] ?? ''
        ];

        if ($request->hasFile('image')) {
            $path = \App\Services\ImageService::uploadAndOptimize($request->file('image'), 'cars');
            $sql .= ", image=?";
            $params[] = asset('storage/' . $path);
        }

        $sql .= " WHERE id=?";
        $params[] = $id;

        $stmt = $conn->prepare($sql);
        $stmt->execute($params);

        return redirect()->route('admin.cars.index')->with('success', 'Car updated successfully.');
    }

    public function destroy($id)
    {
        global $conn;
        $stmt = $conn->prepare("DELETE FROM cars WHERE id = ?");
        $stmt->execute([$id]);
        return redirect()->route('admin.cars.index')->with('success', 'Car deleted successfully.');
    }
}
